var aboutApp = `
Reducing Road Acceidents database app version 18/8/2019

See https://www.bowlandmaths.org.uk/projects/reducing_road_accidents.html for the complete
package of classroom materials and teacher guides of which this app forms a part.

Original design by Daniel Pead and Malcolm Swan at the Shell Centre for Mathematical Education
© 2019 Daniel Pead 
© 2008 Bowland Charitable Trust
All rights reserved

These materials may be freely copied for non-commercial use, provided
that all attributions and license information is retained.

Includes some third-party libraries included under the terms of their own licenses
- see individual folders in js/libs for details.

THIS APP DOES NOT WORK ON INTERNET EXPLORER OR OTHER OLD BROWSERS
Requires ECMAScript 6 or later and HTML5 support
`;
var Accidents;
(function (Accidents) {
    class Msg {
        static send(e) {
            window.setTimeout(() => document.dispatchEvent(e));
        }
        static selectRecord(recordID) {
            const eRecordChange = new CustomEvent("acc.recordchanged", {
                detail: { recordID: recordID }
            });
            Msg.send(eRecordChange);
        }
        static handleSelectRecord(handler) {
            document.addEventListener("acc.recordchanged", handler);
        }
        static dataChanged(recordID) {
            Msg.send(new CustomEvent("acc.datachanged", {
                detail: { recordID: recordID }
            }));
        }
        static handleDataChanged(handler) {
            document.addEventListener("acc.datachanged", handler);
        }
        static chartChanged() {
            Msg.send(new CustomEvent("acc.chartchanged"));
        }
        static handleChartChanged(handler) {
            document.addEventListener("acc.chartchanged", handler);
        }
    }
    Accidents.Msg = Msg;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    let FiltType;
    (function (FiltType) {
        FiltType["key"] = "key";
        FiltType["number"] = "number";
        FiltType["time"] = "time";
        FiltType["date"] = "date";
        FiltType["string"] = "string";
        FiltType["circ"] = "circ";
    })(FiltType = Accidents.FiltType || (Accidents.FiltType = {}));
    class Filter {
        constructor(iField, value, maxVal) {
            this.iField = iField;
            this.filtType = Accidents.dataSet.getFieldDef(iField).type;
            if (value instanceof Array) {
                this.validKeys = [...value];
                this.maxVal = this.minVal = undefined;
                if (this.filtType != FiltType.key)
                    throw new Error("Invalid filter type");
            }
            else {
                if (this.filtType == FiltType.time) {
                    if (typeof value == 'string') {
                        value = Accidents.timeToNumber(value);
                        if (maxVal !== undefined)
                            maxVal = Accidents.timeToNumber(maxVal.toString());
                    }
                    else {
                        value = 0;
                        maxVal = 0;
                    }
                }
                this.minVal = value;
                if (maxVal === undefined) {
                    this.maxVal = value;
                }
                else {
                    this.maxVal = maxVal;
                }
            }
        }
        test(rec) {
            let value = rec[this.iField];
            let match = false;
            if (this.filtType == FiltType.time) {
                value = Accidents.timeToNumber(value);
            }
            if (this.filtType == FiltType.circ || this.filtType == FiltType.time) {
                if (this.minVal >= this.maxVal) {
                    match = (value >= this.minVal || value <= this.maxVal);
                }
                else {
                    match = (value >= this.minVal && value <= this.maxVal);
                }
            }
            else if (this.filtType == FiltType.key) {
                match = (this.validKeys.includes(value.toString()));
            }
            else {
                match = (value >= this.minVal && value <= this.maxVal);
            }
            return match;
        }
        checkParam(param) {
            if (!Filter.setable.includes(param))
                throw new Error(`Filter param '${param}' not settable`);
        }
        jogUp(param) {
            this.checkParam(param);
            const field = Accidents.dataSet.getFieldDef(this.iField);
            if (this.filtType == FiltType.time) {
                this[param] = (this[param] - this[param] % 30 + 30) % 1440;
            }
            else if (field.key) {
                const keys = Object.keys(field.key);
                let i = keys.indexOf(this[param].toString()) + 1;
                if (i >= keys.length) {
                    if (this.filtType == FiltType.circ)
                        i = i % keys.length;
                    else
                        i = keys.length - 1;
                }
                this[param] = keys[i];
            }
            else if (this.filtType == FiltType.number) {
                let n = parseInt(this[param]);
                if (n < field.max)
                    n++;
                this[param] = n;
            }
        }
        jogDown(param) {
            this.checkParam(param);
            const field = Accidents.dataSet.getFieldDef(this.iField);
            if (this.filtType == FiltType.time) {
                this[param] = (this[param] - this[param] % 30 - 30);
                if (this[param] < 0)
                    this[param] = 1440 + this[param];
            }
            else if (field.key) {
                const keys = Object.keys(field.key);
                let i = keys.indexOf(this[param].toString()) - 1;
                if (i < 0) {
                    if (this.filtType == FiltType.circ)
                        i = keys.length - 1;
                    else
                        i = 0;
                }
                this[param] = keys[i];
            }
            else if (this.filtType == FiltType.number) {
                let n = parseInt(this[param]);
                if (n > 0)
                    n--;
                this[param] = n;
            }
        }
        setParam(param, value) {
            this.checkParam(param);
            const field = Accidents.dataSet.getFieldDef(this.iField);
            if (typeof value == "string") {
                if (this.filtType == FiltType.time) {
                    value = Accidents.timeToNumber(value);
                }
                else {
                    value = parseInt(value);
                }
            }
            if (isNaN(value) || value > field.max || value < field.min)
                return;
            this[param] = value;
        }
        getParam(param) {
            this.checkParam(param);
            if (this.filtType == FiltType.time) {
                return Accidents.numberToTime(this[param]);
            }
            else {
                return this[param].toString();
            }
        }
        setValid(val, state) {
            if (this.validKeys === undefined)
                return;
            const i = this.validKeys.indexOf(val);
            if (i < 0) {
                if (state)
                    this.validKeys.push(val);
            }
            else {
                if (!state)
                    this.validKeys.splice(i, 1);
            }
        }
    }
    Filter.setable = ['minVal', 'maxVal'];
    Accidents.Filter = Filter;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    function timeToNumber(s) {
        const parts = /^([0-2]?[0-9]):([0-5][0-9])$/.exec(s);
        if (!parts)
            return 0;
        return (parseInt(parts[1]) * 60 + parseInt(parts[2])) % 1440;
    }
    Accidents.timeToNumber = timeToNumber;
    function zeroPad(n, len = 2) {
        return ('0'.repeat(len) + Math.round(n).toString()).slice(-len);
    }
    function numberToTime(n) {
        const min = n % 60;
        const hr = (n - min) / 60;
        return zeroPad(hr, 2) + ':' + zeroPad(min, 2);
    }
    Accidents.numberToTime = numberToTime;
    class DataSet {
        constructor() {
            this.mapInfo = undefined;
            this.fieldIndex = undefined;
            this.fields = undefined;
            this.records = undefined;
            this.translations = undefined;
            this.dataIndex = null;
            this.currRecID = null;
            this.iCurrRec = null;
            this.nextFiltID = 0;
            this.iSortField = -1;
            this.sortDesc = false;
            this.utcDates = [];
            let data;
            const src = document.getElementById("data_div");
            if (!src)
                throw new Error("data_div not found");
            if (!src.textContent)
                throw new Error("data_div not loaded");
            try {
                data = JSON.parse(src.textContent);
            }
            catch (_a) {
                throw new Error("Error parsing accident data");
            }
            this.filters = new Map();
            this.mapInfo = data['mapInfo'];
            this.fieldIndex = data['fieldIndex'];
            this.fields = data['fields'];
            this.records = data['records'];
            this.translations = data['translations'];
            this.dataIndex = Array.from(this.records.keys());
            this.currRecID = this.dataIndex[0];
            this.iCurrRec = 0;
            for (var f of this.fields) {
                if (f.key) {
                    const k = Object.keys(f.key);
                    f.min = parseInt(k[0]);
                    f.max = parseInt(k.pop());
                }
            }
            for (let d of this.records) {
                let year = 0;
                let dateStr = "";
                let timeStr = "";
                for (let f of this.fields) {
                    if (f.type == 'number') {
                        if (f.max === undefined)
                            f.max = d[f.index];
                        else
                            f.max = Math.max(f.max, d[f.index]);
                        if (f.min === undefined)
                            f.min = d[f.index];
                        else
                            f.min = Math.min(f.min, d[f.index]);
                    }
                    switch (f.name) {
                        case 'year':
                            year = 2004 + d[f.index];
                            break;
                        case 'date':
                            dateStr = d[f.index];
                            break;
                        case 'time':
                            timeStr = d[f.index];
                            break;
                        default:
                            break;
                    }
                }
                let utcDate = new Date(`${dateStr} ${year} ${timeStr}`);
                if (isNaN(utcDate))
                    utcDate = new Date();
                this.utcDates.push(utcDate);
            }
        }
        static getInstance() {
            if (!DataSet.instance)
                DataSet.instance = new DataSet();
            return DataSet.instance;
        }
        *iterateFiltered() {
            for (var i of this.dataIndex) {
                yield [i, this.records[i]];
            }
        }
        getFieldNameDef(fieldName) {
            return this.fields[this.fieldIndex[fieldName]];
        }
        getFieldDef(fieldIndex) {
            return this.fields[fieldIndex];
        }
        getField(id, field) {
            return this.records[id][this.fieldIndex[field]];
        }
        getRecFieldDisplay(rec, iField) {
            const v = rec[iField];
            const k = this.fields[iField].key;
            if (k == undefined) {
                return v.toString();
            }
            else {
                return k[v];
            }
        }
        getRecord(iRecord, display = true) {
            const full = {};
            const rec = this.records[iRecord];
            if (rec === undefined)
                return null;
            for (let fi in this.fields) {
                const f = this.fields[fi];
                if (display && f.key != undefined) {
                    full[f.name] = f.key[rec[fi]];
                }
                else {
                    full[f.name] = rec[fi];
                }
            }
            return full;
        }
        recordCount(recordID) {
            const i = this.dataIndex.indexOf(recordID);
            return [i + 1, this.dataIndex.length];
        }
        prevRecord() {
            if (this.iCurrRec > 0) {
                this.currRecID = this.dataIndex[--this.iCurrRec];
                Accidents.Msg.selectRecord(this.currRecID);
            }
        }
        nextRecord() {
            if (this.iCurrRec + 1 < this.dataIndex.length) {
                this.currRecID = this.dataIndex[++this.iCurrRec];
                Accidents.Msg.selectRecord(this.currRecID);
            }
        }
        selectRecordByID(recordID) {
            if (recordID === undefined)
                recordID = this.dataIndex[0];
            this.iCurrRec = this.dataIndex.indexOf(recordID);
            if (this.iCurrRec >= 0) {
                this.currRecID = recordID;
                Accidents.Msg.selectRecord(this.currRecID);
            }
        }
        addFilter(filter) {
            if (this.filters.size >= DataSet.maxFilters)
                throw new Error("Too many filters");
            this.filters.set(++this.nextFiltID, filter);
            this.filterData();
            return this.nextFiltID;
        }
        removeFilter(iFilter) {
            const ret = this.filters.delete(iFilter);
            this.filterData();
            return ret;
        }
        getFilter(iFilter) {
            return this.filters.get(iFilter);
        }
        filterData() {
            if (this.filters.size == 0) {
                this.dataIndex = Array.from(this.records.keys());
            }
            else {
                this.dataIndex = [];
                for (let iRec in this.records) {
                    let include = true;
                    for (let [iFilt, filt] of this.filters) {
                        include = include && filt.test(this.records[iRec]);
                    }
                    if (include) {
                        this.dataIndex.push(parseInt(iRec));
                    }
                }
            }
            if (!this.dataIndex.length) {
                this.currRecID = undefined;
                this.iCurrRec = -1;
            }
            else {
                if (this.currRecID === undefined || this.dataIndex.length == 1) {
                    this.iCurrRec = 0;
                    this.currRecID = this.dataIndex[0];
                }
                else {
                    let i = 0;
                    while (i < this.dataIndex.length && this.dataIndex[i] < this.currRecID)
                        i++;
                    this.iCurrRec = (i >= this.dataIndex.length) ? 0 : i;
                    this.currRecID = this.dataIndex[this.iCurrRec];
                }
            }
            this.doSort();
            Accidents.Msg.dataChanged(this.currRecID);
        }
        canAddFilter() {
            return this.filters.size < DataSet.maxFilters;
        }
        getSort() {
            return [this.iSortField, this.sortDesc];
        }
        setSort(iField, desc) {
            if (desc === undefined) {
                desc = (this.iSortField == iField) ? (!this.sortDesc) : false;
            }
            this.iSortField = iField;
            this.sortDesc = desc;
            this.doSort();
            Accidents.Msg.dataChanged(this.currRecID);
            return [iField, desc];
        }
        doSort() {
            this.dataIndex.sort((a, b) => this.sortCompare(a, b));
        }
        sortCompare(iRecA, iRecB) {
            if (this.iSortField < 0)
                return iRecA - iRecB;
            const valA = this.records[iRecA][this.iSortField].toString();
            const valB = this.records[iRecB][this.iSortField].toString();
            let ret = 0;
            switch (this.fields[this.iSortField].type) {
                case 'number':
                case 'key':
                case 'circ':
                    ret = parseInt(valA) - parseInt(valB);
                    break;
                case 'time':
                    ret = timeToNumber(valA) - timeToNumber(valB);
                    break;
                case 'date':
                    ret = this.utcDates[iRecA].valueOf() - this.utcDates[iRecB].valueOf();
                    break;
                default:
                    ret = (valA) == (valB) ? 0 : ((valA) < (valB) ? -1 : 1);
                    break;
            }
            ret = Math.sign(ret);
            if (this.sortDesc)
                ret = -ret;
            return ret;
        }
        getFreqTable(fieldIDorName, useFilt = true) {
            let dataIndex = this.dataIndex;
            if (!useFilt) {
                dataIndex = Array.from(this.records.keys());
            }
            let fieldID = parseInt(fieldIDorName);
            if (isNaN(fieldID) && fieldIDorName in this.fieldIndex)
                fieldID = this.fieldIndex[fieldIDorName];
            if (isNaN(fieldID))
                return null;
            const freqTab = new FreqTable(this.fields[fieldID]);
            let total = 0;
            for (let iRec of dataIndex) {
                freqTab.add(this.records[iRec][fieldID]);
            }
            return freqTab;
        }
        getDefaultFieldID(use, name) {
            for (let f of this.fields)
                if (f.use.includes(use)) {
                    if (name === undefined || name == f.name)
                        return f.index;
                }
            return -1;
        }
    }
    DataSet.maxFilters = 4;
    Accidents.DataSet = DataSet;
    class FreqTable {
        constructor(field) {
            this.data = [];
            this.validCases = 0;
            if (field.key) {
                for (let v in field.key) {
                    this.data.push({ index: this.data.length, label: field.key[v], min: parseInt(v), max: parseInt(v), freq: 0 });
                }
            }
            else if (field.type == "number") {
                let binSize = parseInt(field.bin);
                if (isNaN(binSize))
                    binSize = Math.ceil(field.max / 20);
                let nBins = Math.floor(field.max / binSize) + 1;
                for (let i = 0; i < nBins; i++) {
                    const min = i * binSize;
                    const max = (i + 1) * binSize - 1;
                    const label = (binSize == 1) ? `${min}` : `${min}–${max}`;
                    this.data.push({ index: i, label: label, min: min, max: max, freq: 0 });
                }
            }
            else {
                throw new Error("Frequency table requires field of type 'number' or with a key");
            }
            this.maxFreq = 0;
            this.cases = 0;
        }
        *[Symbol.iterator]() {
            for (let f of this.data) {
                yield f;
            }
        }
        add(value) {
            if (typeof value == 'string')
                value = Number(value);
            for (let bin of this.data) {
                if (value >= bin.min && value <= bin.max) {
                    bin.freq++;
                    this.validCases++;
                    this.maxFreq = Math.max(this.maxFreq, bin.freq);
                }
            }
            this.cases++;
        }
        item(i) {
            return this.data[i];
        }
        getMax() {
            return this.maxFreq;
        }
        getLength() {
            return this.data.length;
        }
        sort(col, desc) {
            if (col) {
                this.data.sort((a, b) => {
                    let ret = Math.sign(a.freq - b.freq);
                    return desc ? -ret : ret;
                });
            }
            else {
                this.data.sort((a, b) => {
                    let ret = Math.sign(a.index - b.index);
                    return desc ? -ret : ret;
                });
            }
        }
        makeScale(suggSteps) {
            if (!suggSteps)
                suggSteps = 10;
            if ((this.maxFreq) <= Number.MIN_VALUE) {
                return { max: suggSteps, step: 1, steps: suggSteps };
            }
            var step = null;
            if (!step) {
                let mant = Math.log10(this.maxFreq / suggSteps);
                const power = Math.floor(step);
                mant -= power;
                if (mant <= 0.3)
                    step = Math.pow(10, power);
                else if (mant < 0.69)
                    step = 2 * Math.pow(10, power);
                else
                    step = 5 * Math.pow(10, power);
                if (step < 1)
                    step = 1;
            }
            const steps = Math.ceil(this.maxFreq / step);
            const max = steps * step;
            return { max, step, steps };
        }
    }
    Accidents.FreqTable = FreqTable;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    let optDlgElem;
    class OptionDialog {
        constructor() {
            if (!optDlgElem) {
                optDlgElem = document.querySelector("dialog#option_dialog");
                dialogPolyfill.registerDialog(optDlgElem);
                optDlgElem.querySelector(".close_btn").addEventListener("click", (e) => {
                    optDlgElem.close("");
                });
            }
        }
        show(title, handler, ...rest) {
            this.handler = handler;
            optDlgElem.returnValue = "";
            optDlgElem.querySelector(".dlg_title").textContent = title;
            const form = optDlgElem.querySelector("form");
            this.fill(form, rest);
            this.handlerClosure = (e) => this.dlgClosed(e);
            optDlgElem.addEventListener("close", this.handlerClosure);
            optDlgElem.showModal();
        }
        dlgClosed(e) {
            optDlgElem.removeEventListener("close", this.handlerClosure);
            this.handlerClosure = undefined;
            if (this.handler)
                this.handler(optDlgElem.returnValue);
            this.handler = undefined;
        }
        fill(form, ...rest) {
            form.querySelectorAll(".dlg_option").forEach(elem => elem.remove());
            this.nextTabIndex = 0;
        }
        addButton(form, value, label) {
            this.nextTabIndex++;
            const btn = document.createElement("button");
            btn.setAttribute("value", value);
            btn.setAttribute("tabindex", this.nextTabIndex.toString());
            btn.classList.add("dlg_option");
            btn.textContent = label;
            form.appendChild(btn);
            if (this.nextTabIndex == 1)
                window.setTimeout(() => btn.focus());
            return btn;
        }
    }
    class FieldNamePicker extends OptionDialog {
        constructor() {
            super();
        }
        static getInstance() {
            if (!FieldNamePicker.instance)
                FieldNamePicker.instance = new FieldNamePicker();
            return FieldNamePicker.instance;
        }
        show(title, handler, use) {
            super.show(title, handler, use);
        }
        fill(form, [use]) {
            super.fill(form);
            for (let f of Accidents.dataSet.fields) {
                if (f.use && f.use.includes(use)) {
                    this.addButton(form, f.name, f.title);
                }
            }
        }
    }
    Accidents.FieldNamePicker = FieldNamePicker;
    class FieldValuePicker extends OptionDialog {
        constructor() {
            super();
        }
        static getInstance() {
            if (!FieldValuePicker.instance)
                FieldValuePicker.instance = new FieldValuePicker();
            return FieldValuePicker.instance;
        }
        show(title, handler, field) {
            super.show(title, handler, field);
        }
        fill(form, [field]) {
            super.fill(form);
            const fd = Accidents.dataSet.getFieldNameDef(field);
            for (let [value, label] of Object.entries(fd.key)) {
                this.addButton(form, value, label);
            }
        }
    }
    Accidents.FieldValuePicker = FieldValuePicker;
})(Accidents || (Accidents = {}));
var MinSvg;
(function (MinSvg) {
    class ViewBox {
        constructor(x, y, w, h) {
            if (x instanceof ViewBox) {
                this.x = x.x;
                this.y = x.y;
                this.w = x.w;
                this.h = x.h;
            }
            else if (typeof x == 'number') {
                this.x = x;
                this.y = y;
                this.w = w;
                this.h = h;
            }
            else if (typeof x == 'string') {
                const vals = x.split(' ');
                if (vals.length != 4) {
                    throw new Error("Bad initialiser for Box");
                }
                this.x = Number(vals[0]);
                this.y = Number(vals[1]);
                this.w = Number(vals[2]);
                this.h = Number(vals[3]);
            }
        }
        copy() {
            return new ViewBox(this);
        }
        toString() {
            return `${this.x} ${this.y} ${this.w} ${this.h}`;
        }
        zoomPan(factor, cx, cy, dx, dy) {
            this.w *= factor;
            this.h *= factor;
            cx -= dx;
            cy -= dy;
            this.x = cx - (cx + dx - this.x) * factor;
            this.y = cy - (cy + dy - this.y) * factor;
        }
        getCenter() {
            return [this.x + this.w / 2, this.y + this.h / 2];
        }
        offset(dx, dy) {
            this.x += dx;
            this.y += dy;
        }
    }
    MinSvg.ViewBox = ViewBox;
})(MinSvg || (MinSvg = {}));
var MinSvg;
(function (MinSvg) {
    const svgNs = "http://www.w3.org/2000/svg";
    function deCamelIze(s) {
        return s.replace(/([A-Z])([^A-Z]+)/g, '-$1$2').toLowerCase();
    }
    class SvgContainer {
        constructor(root) {
            this.root = root;
        }
        set(att, value) {
            this.root.setAttribute(att, value.toString());
        }
        newElem(tag, attributes) {
            const e = document.createElementNS(svgNs, tag);
            for (let a in attributes) {
                e.setAttribute(deCamelIze(a), attributes[a]);
            }
            return e;
        }
        appendNewElem(tag, attributes = {}, parent = null) {
            const e = this.newElem(tag, attributes);
            if (!parent)
                parent = this.root;
            parent.appendChild(e);
            return e;
        }
        groupWithID(id, atts = {}, parent) {
            let g = this.appendNewElem("g", atts, parent);
            g.id = id;
            return new SvgContainer(g);
        }
        group(atts = {}, parent) {
            let g = this.appendNewElem("g", atts, parent);
            return new SvgContainer(g);
        }
        line(x1, y1, x2, y2, atts = {}, parent) {
            return this.appendNewElem("line", Object.assign({ x1, y1, x2, y2 }, atts), parent);
        }
        circle(cx, cy, r, atts = {}, parent) {
            return this.appendNewElem("circle", Object.assign({ cx, cy, r }, atts), parent);
        }
        rect(x, y, width, height, atts = {}, parent) {
            return this.appendNewElem("rect", Object.assign({ x, y, width, height }, atts), parent);
        }
        text(x, y, text, atts = {}, parent) {
            const e = this.appendNewElem("text", Object.assign({ x, y }, atts), parent);
            e.textContent = text;
            return e;
        }
        path(d, atts = {}, parent) {
            const e = this.appendNewElem("path", Object.assign({ d }, atts), parent);
            return e;
        }
        use(id, x, y, atts = {}, parent) {
            const e = this.appendNewElem("use", Object.assign({ "xlink:href": '#' + id, x, y }, atts), parent);
            return e;
        }
        removeRoot() {
            if (this.root.parentNode)
                this.root.parentNode.removeChild(this.root);
            this.root = null;
        }
    }
    MinSvg.SvgContainer = SvgContainer;
    class SvgRoot extends SvgContainer {
        constructor(elementID) {
            super(null);
            const e = document.getElementById(elementID);
            if (!e)
                throw new Error(`SVG element '${elementID}' not found`);
            if (e.tagName.toLowerCase() != 'svg')
                throw new Error(`Element '${elementID}' not found`);
            this.root = e;
        }
        setViewBox(vb) {
            this.root.setAttribute("viewBox", vb.toString());
        }
        animateViewBox(vb) {
            const currVb = this.root.getAttribute("viewBox");
            let eAnim = this.root.querySelector(".vb_anim");
            if (!eAnim)
                eAnim = this.appendNewElem("animate", {
                    class: "vb_anim",
                    attributeName: "viewbox", dur: "1s", begin: "indefinite"
                });
            eAnim.setAttribute("values", currVb + '; ' + vb.toString());
            eAnim.beginElement();
        }
        getViewBox() {
            return new MinSvg.ViewBox(this.root.getAttribute("viewBox"));
        }
    }
    MinSvg.SvgRoot = SvgRoot;
})(MinSvg || (MinSvg = {}));
var Accidents;
(function (Accidents) {
    class Pane {
        constructor(selector) {
            this.paneDiv = document.querySelector(selector);
            this.display = this.paneDiv.style.display;
            this.paneDiv.style.display = "none";
            this.visible = false;
        }
        show() {
            this.visible = true;
            this.paneDiv.style.display = this.display;
        }
        hide() {
            this.visible = false;
            this.paneDiv.style.display = "none";
        }
    }
    Accidents.Pane = Pane;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    function qsListen(sel, eventName, handler, root = document) {
        const element = root.querySelector(sel);
        if (!element)
            throw new Error(`Element '${sel}' doesn't exist`);
        element.addEventListener(eventName, handler);
    }
    Accidents.qsListen = qsListen;
    function hasFullScreen() {
        if (document.fullscreenEnabled === undefined) {
            return false;
        }
        else {
            return document.fullscreenEnabled;
        }
    }
    Accidents.hasFullScreen = hasFullScreen;
    function isFullScreen() {
        if (document.fullscreenElement === undefined) {
            return false;
        }
        else {
            return document.fullscreenElement != null;
        }
    }
    Accidents.isFullScreen = isFullScreen;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class MapPage extends Accidents.Pane {
        constructor() {
            super('#map_page');
            this.dataGroup = null;
            this.gridGroup = null;
            this.scale = 1;
            this.cX = 0;
            this.cY = 0;
            this.clientR = null;
            this.mapUnitScale = 1;
            this.currentDataID = undefined;
            this.key = undefined;
            if (Accidents.dataSet.mapInfo.mapUrl.startsWith('#')) {
                this.mapSvg = new MinSvg.SvgRoot(Accidents.dataSet.mapInfo.mapUrl.substr(1));
            }
            else {
                throw new Error("Map loading not supported");
            }
            this.gridSvg = new MinSvg.SvgRoot('grid_svg');
            this.dataSvg = new MinSvg.SvgRoot('data_svg');
            this.vbInit = this.mapSvg.getViewBox();
            this.vbCurrent = this.vbInit.copy();
            this.vbPinching = null;
            this.gridSvg.setViewBox(this.vbCurrent);
            this.dataSvg.setViewBox(this.vbCurrent);
            this.cX = this.vbInit.x + this.vbInit.w / 2;
            this.cY = this.vbInit.y + this.vbInit.h / 2;
            this.getMapData();
            this.onResize(null);
            Accidents.Msg.handleSelectRecord(e => {
                this.currentDataID = e.detail.recordID;
                this.dataSelectionChange();
            });
            Accidents.Msg.handleDataChanged(e => {
                this.currentDataID = e.detail.recordID;
                this.drawData();
                this.dataSelectionChange();
            });
            Accidents.qsListen("#btn_map_zoomout", "click", e => this.onZoomOut(e));
            Accidents.qsListen("#btn_map_zoomin", "click", e => this.onZoomIn(e));
            Accidents.qsListen("#btn_map_home", "click", e => this.onHome(e));
            this.key = document.getElementById("map_key");
            this.key.style.visibility = "hidden";
            Accidents.qsListen("#btn_map_key_toggle", "click", e => this.toggleMapKey());
            Accidents.qsListen("#btn_map_key_close", "click", e => this.toggleMapKey(false));
            this.gestures = new Hammer(this.mapSvg.root, {});
            this.gestures.get("pan").set({ "direction": Hammer.DIRECTION_ALL });
            this.gestures.get('pinch').set({ enable: true });
            this.gestures.on("pan", ev => this.onPan(ev));
            this.gestures.on("pinch", ev => this.onPinch(ev));
            this.gestures.on("pinchend", ev => this.onPinchEnd(ev));
            this.mapSvg.root.addEventListener("wheel", ev => this.onWheel(ev));
            window.addEventListener("resize", ev => this.onResize(ev));
            this.drawGrid(this.vbCurrent, this.scale);
            this.drawData();
        }
        static getInstance() {
            if (!MapPage.instance)
                MapPage.instance = new MapPage();
            return MapPage.instance;
        }
        onResize(ev) {
            this.clientR = this.mapSvg.root.getBoundingClientRect();
        }
        show() {
            super.show();
            this.onResize(null);
        }
        onWheel(ev) {
            const [x, y] = this.clientToMapCoords(ev.clientX, ev.clientY);
            let factor = 1 + 0.004 * ev.deltaY;
            this.zoom(factor, x, y);
            ev.preventDefault();
        }
        toggleMapKey(force) {
            let vis = false;
            if (force !== undefined) {
                vis = false;
            }
            else {
                vis = !(this.key.style.visibility == "visible");
            }
            this.key.style.visibility = vis ? "visible" : "hidden";
            document.getElementById("btn_map_key_toggle").classList.toggle("on", vis);
        }
        getMapData() {
            this.mapUnitScale = Accidents.dataSet.mapInfo.realWidth / this.vbInit.w;
        }
        getNiceGridStep(realW, nDivs = 10) {
            const tstep = realW / nDivs;
            const digits = Math.floor(Math.log10(tstep));
            let step = Math.pow(10, digits);
            if (step * 5 <= tstep)
                step *= 5;
            else if (step * 2 <= tstep)
                step *= 2;
            return step;
        }
        onZoomIn(e) {
            const [x, y] = this.getFocusCoords();
            this.zoom(1 / 2, x, y);
        }
        onZoomOut(e) {
            const [x, y] = this.getFocusCoords();
            this.zoom(2, x, y);
        }
        onHome(e) {
            this.vbCurrent = this.vbInit.copy();
            this.scale = 1;
            [this.cX, this.cY] = this.vbCurrent.getCenter();
            this.animateToView(this.vbInit.copy());
        }
        onPan(ev) {
            if ((Date.now() - this.lastPinchTime) < 300)
                return;
            const [dx, dy] = this.clientToMapDistance(ev.deltaX, ev.deltaY, this.vbInit);
            let vb = this.vbCurrent.copy();
            vb.offset(-dx * this.scale, -dy * this.scale);
            this.setView(vb, ev.isFinal);
        }
        onPinch(ev) {
            if (ev.isFinal && ev.scale == 1)
                return;
            const [x, y] = this.clientToMapCoords(ev.center.x, ev.center.y);
            const [dx, dy] = this.clientToMapDistance(ev.deltaX, ev.deltaY);
            let vb = this.vbCurrent.copy();
            const factor = this.checkFactor(1 / ev.scale);
            vb.zoomPan(factor, x, y, dx, dy);
            vb = this.setView(vb, false);
            this.lastPinchTime = Date.now();
        }
        onPinchEnd(ev) {
            const vb = this.mapSvg.getViewBox();
            this.setView(vb, true);
        }
        clientToMapCoords(clX, clY, vb) {
            if (!vb)
                vb = this.vbCurrent;
            const clR = this.clientR;
            const x = vb.x + (clX - clR.left) * vb.w / (clR.width);
            const y = vb.y + (clY - clR.top) * vb.h / (clR.height);
            return [x, y];
        }
        clientToMapDistance(clX, clY, vb) {
            if (!vb)
                vb = this.vbCurrent;
            const clR = this.clientR;
            clX *= vb.w / (clR.width);
            clY *= vb.h / (clR.height);
            return [clX, clY];
        }
        drawGrid(vb, scale) {
            if (this.gridGroup) {
                this.gridGroup.removeRoot();
                this.gridGroup = null;
            }
            this.gridGroup = this.gridSvg.groupWithID("grid_group", { fill: "none", stroke: "grey", "stroke-width": "thin" });
            const uStep = this.getNiceGridStep(vb.w * this.mapUnitScale, 5);
            const mStep = uStep / (this.mapUnitScale * scale);
            let x0 = mStep - (vb.x / scale) % mStep;
            let xU = (vb.x + x0 * scale) * this.mapUnitScale;
            while (x0 <= this.vbInit.w) {
                this.gridGroup.line(x0, 0, x0, this.vbInit.h);
                this.gridGroup.text(x0, this.vbInit.h, Math.round(xU).toString(), { class: "axis_value xaxis", dy: "0.4em" });
                x0 += mStep;
                xU += uStep;
            }
            let y0 = ((this.vbInit.h - vb.y) / scale % mStep);
            let yU = (this.vbInit.h - (y0 * scale + vb.y)) * this.mapUnitScale;
            while (y0 <= this.vbInit.h) {
                this.gridGroup.line(0, y0, this.vbInit.w, y0);
                this.gridGroup.text(0, y0, Math.round(yU).toString(), { class: "axis_value yaxis", dx: "-0.4em" });
                y0 += mStep;
                yU -= uStep;
            }
        }
        drawData() {
            if (this.dataGroup) {
                this.dataGroup.removeRoot();
                this.dataGroup = null;
            }
            this.dataGroup = this.dataSvg.groupWithID("data_group", { fill: "red", stroke: "black", "stroke-width": "2px" });
            const iEast = Number(Accidents.dataSet.fieldIndex['east']);
            const iNorth = Number(Accidents.dataSet.fieldIndex['north']);
            for (const [id, rec] of Accidents.dataSet.iterateFiltered()) {
                const sel = id == this.currentDataID;
                const x = this.vbInit.x + rec[iEast] / this.mapUnitScale;
                const y = this.vbInit.y + this.vbInit.h - rec[iNorth] / this.mapUnitScale;
                const mark = this.dataGroup.circle(x, y, 10, { class: "data_point" });
                mark.setAttribute("r", sel ? "1.5%" : "1%");
                mark.classList.toggle("selected", sel);
                mark.classList.toggle("unselected", !sel);
                mark.id = `dpt_${id}`;
                mark.addEventListener("click", (e) => this.onDataClick(e, id));
            }
            this.dataGroup.set("stroke-width", (2 * this.scale).toString());
        }
        dataSelectionChange() {
            const dataID = `dpt_${this.currentDataID}`;
            document.querySelectorAll(".data_point").forEach((e) => {
                const sel = e.id == dataID;
                e.classList.toggle("selected", sel);
                e.classList.toggle("unselected", !sel);
                e.setAttribute("r", sel ? "1.5%" : "1%");
            });
            this.moveCentre();
        }
        getFocusCoords() {
            if (this.currentDataID == undefined || this.currentDataID < 0)
                return [this.cX, this.cY];
            const east = Accidents.dataSet.getField(this.currentDataID, "east");
            const north = Accidents.dataSet.getField(this.currentDataID, "north");
            const x = this.vbInit.x + east / this.mapUnitScale;
            const y = this.vbInit.y + this.vbInit.h - north / this.mapUnitScale;
            return [x, y];
        }
        moveCentre() {
            const [x, y] = this.getFocusCoords();
            let vb = this.vbCurrent.copy();
            vb.offset(x - this.cX, y - this.cY);
            this.animateToView(vb);
        }
        onDataClick(e, dataID) {
            e.preventDefault();
            e.stopPropagation();
            Accidents.dataSet.selectRecordByID(dataID);
        }
        checkFactor(factor) {
            const newSc = this.scale * factor;
            if (newSc > 1)
                factor = 1 / this.scale;
            if (newSc < 0.1)
                factor = 0.1 / this.scale;
            return factor;
        }
        zoom(factor, x, y) {
            factor = this.checkFactor(factor);
            const vb = this.vbCurrent.copy();
            vb.zoomPan(factor, x, y, 0, 0);
            this.animateToView(vb);
        }
        adjustViewBox(vb) {
            if (vb.w > this.vbInit.w || vb.h > this.vbInit.h || vb.w < 10 || vb.h < 10) {
                vb = this.vbInit.copy();
            }
            else {
                vb = vb.copy();
                if (vb.x > this.vbInit.x + this.vbInit.w - vb.w) {
                    vb.x = this.vbInit.x + this.vbInit.w - vb.w;
                }
                else if (vb.x < this.vbInit.x) {
                    vb.x = this.vbInit.x;
                }
                if (vb.y > this.vbInit.y + this.vbInit.h - vb.h) {
                    vb.y = this.vbInit.y + this.vbInit.h - vb.h;
                }
                else if (vb.y < this.vbInit.y) {
                    vb.y = this.vbInit.y;
                }
            }
            return vb;
        }
        animateToView(vb) {
            if (Accidents.noAnimation) {
                this.setView(vb, true);
                return;
            }
            vb = this.adjustViewBox(vb);
            if (this.animateInterval) {
                clearInterval(this.animateInterval);
                this.animateInterval = 0;
                this.animateSteps = 0;
            }
            let steps = this.animateSteps = 15;
            const vbTarget = vb;
            const deltaw = (vb.w - this.vbCurrent.w) / steps;
            const vbDelta = new MinSvg.ViewBox((vb.x - this.vbCurrent.x) / steps, (vb.y - this.vbCurrent.y) / steps, deltaw, deltaw * this.vbInit.h / this.vbInit.w);
            this.animateInterval = setInterval(() => this.animateVb(vbTarget, vbDelta), 80);
            return vb;
        }
        setView(vb, final) {
            vb = this.adjustViewBox(vb);
            if (this.animateInterval) {
                clearInterval(this.animateInterval);
                this.animateInterval = 0;
                this.animateSteps = 0;
            }
            this.mapSvg.setViewBox(vb);
            this.dataSvg.setViewBox(vb);
            this.drawGrid(vb, vb.w / this.vbInit.w);
            if (final) {
                this.vbCurrent = vb;
                this.scale = vb.w / this.vbInit.w;
                [this.cX, this.cY] = vb.getCenter();
                this.drawData();
            }
            return vb;
        }
        animateVb(vbTarget, vbDelta) {
            let vb = this.vbCurrent;
            vb.offset(vbDelta.x, vbDelta.y);
            vb.w += vbDelta.w;
            vb.h += vbDelta.h;
            if ((this.animateSteps--) > 0
                && Math.abs(vb.x - vbTarget.x) < 1
                && Math.abs(vb.y - vbTarget.y) < 1
                && Math.abs(vb.w - vbTarget.w) < 1
                && Math.abs(vb.h - vbTarget.h) < 1) {
                clearInterval(this.animateInterval);
                this.animateInterval = 0;
                this.animateSteps = 0;
                vb = vbTarget;
            }
            this.mapSvg.setViewBox(vb);
            this.dataSvg.setViewBox(vb);
            this.vbCurrent = vb.copy();
            this.scale = vb.w / this.vbInit.w;
            [this.cX, this.cY] = vb.getCenter();
            this.drawData();
            this.drawGrid(vb, vb.w / this.vbInit.w);
        }
    }
    Accidents.MapPage = MapPage;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    const filterRowTemplates = {};
    let filterPaneDiv;
    let filterRowsDiv;
    let filterRows = new Map();
    class FilterRow {
        constructor(field) {
            this.iFilter = -1;
            this.field = field;
            this.listeners = [];
            this.iFilter = Accidents.dataSet.addFilter(this.createFilter());
            filterRows.set(this.iFilter, this);
            this.buildDisplay();
            this.updateDisplay();
            this.addListener(".btn_filter_delete", "click", (e) => this.remove());
            filterRowsDiv.appendChild(this.div);
        }
        buildDisplay() {
            if (!(this.field.template in filterRowTemplates))
                throw new Error("Unknown template " + this.field.template);
            this.div = filterRowTemplates[this.field.template].cloneNode(true);
        }
        remove() {
            while (this.listeners.length) {
                const [element, event, handler] = this.listeners.pop();
                element.removeEventListener(event, handler);
            }
            this.div.remove();
            Accidents.dataSet.removeFilter(this.iFilter);
            Accidents.filterPane.showHideAddBtn();
            console.log(filterRows, filterRows.size);
            filterRows.delete(this.iFilter);
            if (filterRows.size == 0) {
                Accidents.filterPane.hide();
            }
        }
        getFilter() {
            return Accidents.dataSet.getFilter(this.iFilter);
        }
        updateDisplay() {
            this.div.querySelector(".filter_field_name").textContent = this.field.title;
        }
        filterChanged() {
            this.updateDisplay();
            Accidents.dataSet.filterData();
        }
        addListener(element, event, handler) {
            if (typeof element == 'string') {
                element = this.div.querySelector(element);
            }
            if (!element)
                return;
            element.addEventListener(event, handler);
            this.listeners.push([element, event, handler]);
        }
    }
    class FilterWithJog extends FilterRow {
        constructor(field) {
            super(field);
        }
        buildDisplay() {
            super.buildDisplay();
            this.div.querySelectorAll('.jog').forEach(element => {
                this.addListener(element, "click", (e) => this.onJog(e));
            });
        }
        onJog(e) {
            const filt = this.getFilter();
            const param = e.currentTarget.name;
            if (e.currentTarget.classList.contains("jog_up")) {
                filt.jogUp(param);
            }
            else {
                filt.jogDown(param);
            }
            this.filterChanged();
        }
    }
    class FilterKeyRange extends FilterWithJog {
        constructor(field) {
            super(field);
        }
        buildDisplay() {
            super.buildDisplay();
            this.addListener(".jog_value[name='minVal']", "click", (e) => this.chooseValue(e.currentTarget.name));
            this.addListener(".jog_value[name='maxVal']", "click", (e) => this.chooseValue(e.currentTarget.name));
        }
        updateDisplay() {
            super.updateDisplay();
            this.div.querySelector(`.jog_value[name='minVal']`).textContent = this.field.key[this.getFilter().minVal];
            this.div.querySelector(`.jog_value[name='maxVal']`).textContent = this.field.key[this.getFilter().maxVal];
        }
        chooseValue(which) {
            Accidents.fieldValuePicker.show(Accidents.dataSet.translations["selectvalue"], (value) => {
                this.getFilter().setParam(which, value);
                this.filterChanged();
            }, this.field.name);
        }
        createFilter() {
            return new Accidents.Filter(this.field.index, this.field.min, this.field.max);
        }
    }
    class FilterNumberRange extends FilterWithJog {
        constructor(field) {
            super(field);
        }
        buildDisplay() {
            super.buildDisplay();
            this.div.querySelectorAll("input.jog_value").forEach((inp) => {
                inp.min = this.field.min.toString();
                inp.max = this.field.max.toString();
                this.addListener(inp, "change", e => this.changed(e));
            });
        }
        updateDisplay() {
            super.updateDisplay();
            this.div.querySelectorAll("input.jog_value").forEach((inp) => {
                inp.value = this.getFilter().getParam(inp.name).toString();
            });
        }
        changed(e) {
            const el = e.currentTarget;
            this.getFilter().setParam(el.name, el.value);
            this.filterChanged();
        }
        createFilter() {
            return new Accidents.Filter(this.field.index, this.field.min, this.field.max);
        }
    }
    class FilterNumber extends FilterNumberRange {
        constructor(field) {
            super(field);
        }
        filterChanged() {
            this.getFilter().maxVal = this.getFilter().minVal;
            super.filterChanged();
        }
        changed(e) {
            const el = e.currentTarget;
            this.getFilter().setParam("minVal", el.value);
            this.filterChanged();
        }
        createFilter() {
            return new Accidents.Filter(this.field.index, this.field.min);
        }
    }
    class FilterCheckbox extends FilterRow {
        constructor(field) {
            super(field);
        }
        buildDisplay() {
            super.buildDisplay();
            const cbTemp = this.div.querySelector('.ctl_checkbox');
            this.div.removeChild(cbTemp);
            for (let [k, label] of Object.entries(this.field.key)) {
                const cbSpan = cbTemp.cloneNode(true);
                cbSpan.querySelector('label span.ctl_value_name').textContent = label;
                const cb = cbSpan.querySelector('input');
                if (cb) {
                    cb.setAttribute("value", k.toString());
                    this.addListener(cb, "change", (e) => this.checkChanged(e));
                }
                this.div.appendChild(cbSpan);
            }
        }
        updateDisplay() {
            super.updateDisplay();
            const validKeys = this.getFilter().validKeys;
            for (let k in this.field.key) {
                const cb = this.div.querySelector(`input[value='${k}']`);
                if (cb) {
                    cb.checked = validKeys.indexOf(k) >= 0;
                }
            }
        }
        checkChanged(e) {
            const cb = e.currentTarget;
            this.getFilter().setValid(cb.value, cb.checked);
            this.filterChanged();
        }
        createFilter() {
            return new Accidents.Filter(this.field.index, Object.keys(this.field.key));
        }
    }
    class FilterTimeRange extends FilterWithJog {
        constructor(field) {
            super(field);
        }
        buildDisplay() {
            super.buildDisplay();
            this.div.querySelectorAll("input.jog_value").forEach((inp) => {
                this.addListener(inp, "change", e => this.changed(e));
            });
        }
        updateDisplay() {
            super.updateDisplay();
            this.div.querySelectorAll("input.jog_value").forEach((inp) => {
                inp.value = this.getFilter().getParam(inp.name);
            });
        }
        changed(e) {
            const el = e.currentTarget;
            this.getFilter().setParam(el.name, el.value);
            this.filterChanged();
        }
        createFilter() {
            return new Accidents.Filter(this.field.index, this.field.min, this.field.max);
        }
    }
    let filterConstruct = new Map([
        ["filter_key_range", FilterKeyRange],
        ["filter_number_range", FilterNumberRange],
        ["filter_time_range", FilterTimeRange],
        ["filter_number", FilterNumber],
        ["filter_checkbox", FilterCheckbox]
    ]);
    class FilterPane {
        constructor() {
            this.isVisible = false;
            filterPaneDiv = document.querySelector('#filter_pane');
            filterRowsDiv = filterPaneDiv.querySelector('.filter_rows');
            filterPaneDiv.querySelectorAll("[data-template]").forEach((element) => {
                filterRowTemplates[element.dataset.template] = element;
                filterRowsDiv.removeChild(element);
            });
            this.display = filterPaneDiv.style.display;
            filterPaneDiv.style.display = "none";
            const addBtn = filterPaneDiv.querySelector('#btn_add_filter');
            addBtn.addEventListener("click", e => this.chooseNewFilter());
            addBtn.disabled = true;
        }
        static getInstance() {
            if (!FilterPane.instance)
                FilterPane.instance = new FilterPane();
            return FilterPane.instance;
        }
        chooseNewFilter() {
            Accidents.fieldNamePicker.show(Accidents.dataSet.translations["selectrecordsby"], name => this.addFilter(name), "filter");
        }
        addFilter(name) {
            if (name) {
                const field = Accidents.dataSet.getFieldNameDef(name);
                if (filterConstruct.has(field.template)) {
                    let filtRow = new (filterConstruct.get(field.template))(field);
                }
            }
            this.showHideAddBtn();
        }
        showHideAddBtn() {
            const addBtn = filterPaneDiv.querySelector('#btn_add_filter');
            const canAdd = Accidents.dataSet.canAddFilter();
            addBtn.disabled = !canAdd;
        }
        show() {
            filterPaneDiv.style.display = this.display;
            this.showHideAddBtn();
            if (Accidents.dataSet.canAddFilter())
                this.chooseNewFilter();
            this.isVisible = true;
            document.querySelector("#btn_filter_pane").classList.toggle("on", this.isVisible);
        }
        hide() {
            filterPaneDiv.style.display = "none";
            let f;
            let rows = filterRows.keys();
            while (f in rows)
                filterRows.get(f).remove();
            this.isVisible = false;
            document.querySelector("#btn_filter_pane").classList.toggle("on", this.isVisible);
        }
        toggle() {
            if (this.isVisible)
                this.hide();
            else
                this.show();
            return this.isVisible;
        }
    }
    Accidents.FilterPane = FilterPane;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class TablePage extends Accidents.Pane {
        constructor() {
            super("#table_page");
            this.build();
            Accidents.Msg.handleDataChanged((e) => {
                if (this.visible)
                    this.update();
            });
        }
        static getInstance() {
            if (!TablePage.instance)
                TablePage.instance = new TablePage();
            return TablePage.instance;
        }
        build() {
            this.table = this.paneDiv.querySelector('table');
            const th = this.table.querySelector('thead th');
            const thr = th.parentElement;
            th.remove();
            this.iFields = [];
            for (let field of Accidents.dataSet.fields) {
                if (!field.use.includes("table"))
                    continue;
                let newth = th.cloneNode(true);
                newth.querySelector('.value').textContent = field.short ? field.short : field.title;
                newth.addEventListener("click", (e) => this.toggleSort(e, field.index));
                this.iFields.push(field.index);
                thr.appendChild(newth);
            }
            this.tbody = this.table.querySelector('tbody');
            this.trow = this.tbody.querySelector('tr');
            this.trow.remove();
            const td = this.trow.querySelector('td');
            td.remove();
            for (let i in this.iFields) {
                const newtd = td.cloneNode(true);
                newtd.dataset.iField = this.iFields[i].toString();
                this.trow.appendChild(newtd);
            }
            this.update();
        }
        show() {
            super.show();
            this.update();
        }
        updateRow(row, rec) {
            let col = row.firstChild;
            row.querySelectorAll("td").forEach((col) => {
                if ((!col.dataset) || col.dataset.iField === undefined)
                    return;
                const val = Accidents.dataSet.getRecFieldDisplay(rec, parseInt(col.dataset.iField));
                col.querySelector('.value').textContent = val;
            });
        }
        update() {
            let rows = this.tbody.querySelectorAll("tr");
            let iRow = 0;
            let row;
            for (const [id, rec] of Accidents.dataSet.iterateFiltered()) {
                if (iRow < rows.length) {
                    row = rows.item(iRow);
                    iRow++;
                }
                else {
                    row = this.trow.cloneNode(true);
                    this.tbody.appendChild(row);
                }
                this.updateRow(row, rec);
            }
            while (iRow < rows.length) {
                rows.item(iRow++).remove();
            }
        }
        toggleSort(e, iField) {
            const [i, desc] = Accidents.dataSet.setSort(iField);
            this.table.querySelectorAll("thead th").forEach((elem) => {
                elem.classList.remove("sortasc");
                elem.classList.remove("sortdesc");
            });
            const t = e.currentTarget;
            t.classList.toggle("sortasc", !desc);
            t.classList.toggle("sortdesc", desc);
        }
    }
    Accidents.TablePage = TablePage;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class ChartDataPane extends Accidents.Pane {
        constructor() {
            super("#chart_data_pane");
            this.sortCol = 0;
            this.sortDesc = false;
            this.lastFieldID = -1;
            this.fieldID = -1;
            this.table = this.paneDiv.querySelector('table');
            this.tbody = this.table.querySelector('tbody');
            if (this.tRowTemplate == undefined) {
                this.tRowTemplate = this.tbody.querySelector('tr');
                this.tRowTemplate.remove();
            }
            this.table.querySelector('thead th:first-of-type').addEventListener("click", (e) => this.toggleSort(0));
            this.table.querySelector('thead th:last-of-type').addEventListener("click", (e) => this.toggleSort(1));
            this.paneDiv.querySelector(".btn_choose_variable").addEventListener("click", e => {
                console.log(Accidents.dataSet.translations);
                Accidents.fieldNamePicker.show(Accidents.dataSet.translations["choosevariable"], (name) => this.setVariable(name), "chart");
            });
            Accidents.Msg.handleDataChanged((e) => {
                if (this.visible) {
                    this.getFreq();
                    this.update();
                    this.chartChanged();
                }
            });
            this.getFreq();
            this.update();
        }
        static getInstance() {
            if (!ChartDataPane.instance)
                ChartDataPane.instance = new ChartDataPane();
            return ChartDataPane.instance;
        }
        show() {
            this.getFreq();
            this.update();
            this.chartChanged();
            super.show();
        }
        chartChanged() {
            Accidents.Msg.chartChanged();
        }
        updateRow(row, rec) {
            row.querySelector("span.colour_key").setAttribute("class", `colour_key fill_${rec.index % 16}`);
            row.querySelector("span.label").textContent = rec.label;
            row.querySelector("span.frequency").textContent = rec.freq.toString();
        }
        getFreq() {
            if (this.fieldID < 0)
                this.fieldID = Accidents.dataSet.getDefaultFieldID("chart");
            this.field = Accidents.dataSet.getFieldDef(this.fieldID);
            this.freqTab = Accidents.dataSet.getFreqTable(this.fieldID, true);
            const freqTabNf = Accidents.dataSet.getFreqTable(this.fieldID, false);
            this.freqScale = freqTabNf.makeScale(10);
            if (this.lastFieldID == this.fieldID)
                this.freqTab.sort(this.sortCol, this.sortDesc);
            this.lastFieldID = this.fieldID;
        }
        getVariableTitle(short = false) {
            return (short && this.field.short) ? this.field.short : this.field.title;
        }
        update() {
            document.querySelectorAll(".chart_variable_name").forEach((el) => {
                el.textContent = this.field.title;
            });
            let rows = this.tbody.querySelectorAll("tr");
            let iRow = 0;
            let row;
            for (let ftr of this.freqTab) {
                if (iRow < rows.length) {
                    row = rows.item(iRow);
                    iRow++;
                }
                else {
                    row = this.tRowTemplate.cloneNode(true);
                    this.tbody.appendChild(row);
                }
                this.updateRow(row, ftr);
            }
            while (iRow < rows.length) {
                rows.item(iRow++).remove();
            }
            this.showSortState();
        }
        showSortState() {
            this.table.querySelectorAll("thead th").forEach((elem, i) => {
                elem.classList.toggle("sortasc", i == this.sortCol && !this.sortDesc);
                elem.classList.toggle("sortdesc", i == this.sortCol && this.sortDesc);
            });
        }
        toggleSort(col) {
            this.sortDesc = (this.sortCol == col) ? (!this.sortDesc) : false;
            this.sortCol = col;
            this.freqTab.sort(this.sortCol, this.sortDesc);
            this.update();
            this.chartChanged();
        }
        setVariable(name) {
            if (!name)
                return;
            this.fieldID = Accidents.dataSet.getFieldNameDef(name).index;
            this.getFreq();
            this.update();
            this.chartChanged();
        }
        isContinuous() {
            return ((!this.field.key) && this.sortCol == 0);
        }
    }
    Accidents.ChartDataPane = ChartDataPane;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class ChartPage extends Accidents.Pane {
        constructor(selector) {
            super(selector);
            this.chartSVG = new MinSvg.SvgRoot(this.paneDiv.id + '_svg');
            this.viewBox = this.chartSVG.getViewBox();
            Accidents.Msg.handleChartChanged(() => this._draw());
        }
        show() {
            Accidents.chartDataPane.show();
            super.show();
        }
        hide() {
            Accidents.chartDataPane.hide();
            super.hide();
        }
        _draw() {
            if (this.rootG)
                this.rootG.removeRoot();
            this.rootG = this.chartSVG.groupWithID(this.paneDiv.id + '_root');
            this.draw(this.rootG);
        }
        ;
    }
    Accidents.ChartPage = ChartPage;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class BarPage extends Accidents.ChartPage {
        constructor() {
            super("#bar_page");
            this.getDimensions();
        }
        static getInstance() {
            if (!BarPage.instance)
                BarPage.instance = new BarPage();
            return BarPage.instance;
        }
        getDimensions() {
            this.titleFs = this.viewBox.w / 35;
            this.scaleFs = this.viewBox.w / 50;
            this.pad = this.viewBox.w / 160;
            this.scaleFw = this.scaleFs * 0.56;
            this.xTitleMarg = this.titleFs;
            this.yTitleMarg = this.xTitleMarg;
            this.yMax = this.viewBox.h - this.yTitleMarg - this.scaleFs - this.pad;
        }
        draw(rootG) {
            const vScaleW = this.scaleFw * Math.floor(Math.log10(Accidents.chartDataPane.freqScale.max) + 2);
            const xMin = this.xTitleMarg + vScaleW + 2 * this.pad;
            let g;
            g = rootG.groupWithID("bar_axis_titles", { fontSize: this.titleFs });
            g.text(this.viewBox.w / 2, this.viewBox.h - this.yTitleMarg + this.titleFs, Accidents.chartDataPane.getVariableTitle(), {
                class: "axis_title chart_variable_name",
            });
            const [tx, ty] = [this.xTitleMarg, this.viewBox.h / 2];
            g.text(tx, ty, Accidents.dataSet.mapInfo.freqTitle, {
                class: "axis_title vaxis_title",
                transform: `rotate(-90,${tx},${ty})`
            });
            g = rootG.groupWithID("bar_chart_frame");
            g.rect(xMin, 0, this.viewBox.w - xMin, this.yMax, { class: "frame" });
            let vScale = rootG.groupWithID("bar_chart_vscale", {
                class: "axis_vscale",
                fontSize: this.scaleFs
            });
            let hScale = rootG.groupWithID("bar_chart_hscale", {
                class: "axis_hscale",
                fontSize: this.scaleFs
            });
            let grid = rootG.groupWithID("bar_chart_axis_grid", { class: "gridline" });
            let bars = rootG.groupWithID("bar_chart_axis_bars", { class: "bar_chart_bars" });
            const dy = this.yMax / (Math.max(Accidents.chartDataPane.freqScale.steps, 1));
            for (let i = 0; i <= Accidents.chartDataPane.freqScale.steps; i++) {
                const y = this.yMax - dy * i;
                if (i > 0 && i < Accidents.chartDataPane.freqScale.steps)
                    grid.line(xMin, y, this.viewBox.w, y);
                vScale.text(xMin - this.pad, y, (Accidents.chartDataPane.freqScale.step * i).toString(), {
                    dominantBaseline: "central"
                });
            }
            const nBars = Accidents.chartDataPane.freqTab.getLength();
            const bp = Accidents.chartDataPane.isContinuous() ? 0 : this.pad / 2;
            const dx = (this.viewBox.w - xMin) / Math.max(nBars, 1);
            const barW = dx - 2 * bp;
            const yScale = this.yMax / Accidents.chartDataPane.freqScale.max;
            const hScaleY = this.yMax + this.scaleFs;
            if (Accidents.chartDataPane.isContinuous()) {
                hScale.text(xMin, hScaleY, Accidents.chartDataPane.freqTab.item(0).min.toString());
            }
            for (let i = 0; i < nBars; i++) {
                const yh = Accidents.chartDataPane.freqTab.item(i).freq * yScale;
                let x = xMin + i * dx;
                bars.rect(x + bp, this.yMax - yh, barW, yh, {
                    class: `fill_${Accidents.chartDataPane.freqTab.item(i).index % 16}`
                });
                x += dx;
                if ((i + 1) < nBars)
                    grid.line(x, 0, x, this.yMax);
                if (Accidents.chartDataPane.isContinuous()) {
                    hScale.text(x, hScaleY, (Accidents.chartDataPane.freqTab.item(i).max + 1).toString());
                }
                else {
                    hScale.text(x - dx / 2, hScaleY, Accidents.chartDataPane.freqTab.item(i).label);
                }
            }
        }
    }
    Accidents.BarPage = BarPage;
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class PiePage extends Accidents.ChartPage {
        constructor() {
            super("#pie_page");
            this.deg2r = Math.PI / 180;
            this.getDimensions();
        }
        static getInstance() {
            if (!PiePage.instance)
                PiePage.instance = new PiePage();
            return PiePage.instance;
        }
        getDimensions() {
            const scaleFs = this.viewBox.w / 50;
            this.pieX0 = this.viewBox.x + this.viewBox.w / 2;
            this.pieY0 = this.viewBox.y + this.viewBox.h / 2;
            this.pieR = this.viewBox.h * 0.45;
            this.piePath = [
                `M ${this.pieX0} ${this.pieY0}`,
                `L ${this.pieX0} ${this.pieY0 - this.pieR}`,
                `A ${this.pieR} ${this.pieR} 0 0 1 ${this.pieX0 + this.pieR} ${this.pieY0}`,
                `L ${this.pieX0} ${this.pieY0}`
            ];
        }
        pieSeg(root, a, a1, c) {
            const a0 = a * this.deg2r;
            const x1 = this.pieX0 + this.pieR * Math.sin(a0);
            const y1 = this.pieY0 - this.pieR * Math.cos(a0);
            const f = a > 180 ? "1 1" : "0 1";
            this.piePath[2] = `A ${this.pieR} ${this.pieR} 0 ${f} ${x1} ${y1}`;
            const g = root.group({ class: `fill_${c % 16}`, transform: `rotate(${a1},${this.pieX0},${this.pieY0})` });
            g.path(this.piePath.join(' '));
        }
        pieLabel(root, a, a1, label, rr = 0.5) {
            const a0 = (a1 + a / 2) * this.deg2r;
            const r = this.pieR * rr;
            const x1 = this.pieX0 + r * Math.sin(a0);
            const y1 = this.pieY0 - r * Math.cos(a0);
            root.text(x1, y1, label, { class: "pie_seg_label", dominantBaseline: "central" });
        }
        draw(rootG) {
            const gPie = rootG.group({ class: "pie_disc" });
            const gLab = rootG.group({ class: "pie_labels", fontSize: this.labelFs });
            if (Accidents.chartDataPane.freqTab.validCases) {
                const nSegs = Accidents.chartDataPane.freqTab.getLength();
                const aScale = 360 / Accidents.chartDataPane.freqTab.validCases;
                let a0 = 0;
                let offset = 0.6;
                for (let i = 0; i < nSegs; i++) {
                    const a = nSegs == 1 ? 359 : Accidents.chartDataPane.freqTab.item(i).freq * aScale;
                    if (a > 0.1)
                        this.pieSeg(gPie, a, a0, Accidents.chartDataPane.freqTab.item(i).index);
                    if (a > 1) {
                        if (a < 10 && ((a > 135 && a < 225) || a < 45 || a > 315)) {
                            offset += 0.1;
                            if (offset > 0.9)
                                offset = 0.6;
                        }
                        else
                            offset = 0.6;
                        this.pieLabel(gLab, a, a0, Accidents.chartDataPane.freqTab.item(i).label, offset);
                    }
                    a0 += a;
                }
            }
        }
    }
    Accidents.PiePage = PiePage;
})(Accidents || (Accidents = {}));
window.addEventListener("load", () => Accidents.start());
var Accidents;
(function (Accidents) {
    Accidents.dataSet = undefined;
    Accidents.fieldNamePicker = undefined;
    Accidents.fieldValuePicker = undefined;
    Accidents.reportPane = undefined;
    Accidents.reportSelPane = undefined;
    Accidents.filterPane = undefined;
    Accidents.chartDataPane = undefined;
    Accidents.noAnimation = false;
    Accidents.highContrast = false;
    Accidents.pageConstruct = new Map([
        ["map_page", Accidents.MapPage.getInstance],
        ["table_page", Accidents.TablePage.getInstance],
        ["bar_page", Accidents.BarPage.getInstance],
        ["pie_page", Accidents.PiePage.getInstance]
    ]);
    Accidents.pages = new Map();
    function start() {
        let search = new URL(window.location).searchParams;
        if (search.get('noanim') !== null)
            setNoAnim(true);
        else {
            let mq = window.matchMedia('(prefers-reduced-motion: reduce)');
            setNoAnim(mq && mq.matches);
            if (mq && mq.addEventListener)
                mq.addEventListener("change", e => setNoAnim(e.matches));
        }
        if (search.get('high') !== null)
            setHighContrast(true);
        else {
            let mq = window.matchMedia('(prefers-contrast: high)');
            setHighContrast(mq && mq.matches);
            if (mq && mq.addEventListener)
                mq.addEventListener("change", e => setHighContrast(e.matches));
        }
        Accidents.dataSet = Accidents.DataSet.getInstance();
        Accidents.reportPane = Accidents.ReportPane.getInstance();
        Accidents.reportSelPane = Accidents.ReportSelPane.getInstance();
        Accidents.filterPane = Accidents.FilterPane.getInstance();
        Accidents.chartDataPane = Accidents.ChartDataPane.getInstance();
        Accidents.fieldNamePicker = Accidents.FieldNamePicker.getInstance();
        Accidents.fieldValuePicker = Accidents.FieldValuePicker.getInstance();
        for (let [name, create] of Accidents.pageConstruct) {
            const page = create();
            Accidents.qsListen(`#btn_${name}`, "click", () => {
                selectPage(name);
            });
            Accidents.pages.set(name, page);
        }
        document.querySelector("#btn_filter_pane").addEventListener("click", (e) => {
            hideAbout();
            Accidents.filterPane.toggle();
        });
        document.querySelectorAll('.hideuntilstarted').forEach((element) => {
            element.classList.remove("hideuntilstarted");
        });
        document.querySelectorAll(".freqTitle").forEach(e => {
            e.textContent = Accidents.dataSet.mapInfo.freqTitle;
        });
        createFullScreenButton();
        selectPage("map_page");
        Accidents.dataSet.selectRecordByID();
    }
    Accidents.start = start;
    function selectPage(newName) {
        for (let [name, page] of Accidents.pages) {
            if (newName != name) {
                page.hide();
                document.querySelector(`#btn_${name}`).classList.toggle("on", false);
            }
        }
        for (let [name, page] of Accidents.pages) {
            if (newName == name) {
                page.show();
                document.querySelector(`#btn_${name}`).classList.toggle("on", true);
            }
        }
        document.body.setAttribute("data-mode", newName);
        hideAbout();
    }
    function createFullScreenButton() {
        const fsb = document.querySelector("button.fullscreen");
        const fs = Accidents.isFullScreen();
        if (Accidents.hasFullScreen()) {
            fsb.style.visibility = "visible";
            fsb.addEventListener("click", (e) => {
                if (Accidents.isFullScreen()) {
                    e.currentTarget.classList.toggle("on", false);
                    document.exitFullscreen();
                }
                else {
                    e.currentTarget.classList.toggle("on", true);
                    document.documentElement.requestFullscreen();
                }
            });
        }
        else {
            fsb.style.visibility = "hidden";
        }
    }
    let showAbout = 2;
    function hideAbout() {
        if (showAbout > 0) {
            showAbout--;
            if (showAbout == 0)
                document.getElementById("about_pane").style.display = "none";
        }
    }
    function setHighContrast(state = true) {
        Accidents.highContrast = state;
        document.body.classList.toggle("high", state);
    }
    function setNoAnim(state = true) {
        Accidents.noAnimation = state;
        document.body.classList.toggle("noanim", state);
    }
})(Accidents || (Accidents = {}));
var Accidents;
(function (Accidents) {
    class DataPane {
        constructor(selector) {
            this.rootEl = document.querySelector(selector);
            Accidents.Msg.handleSelectRecord(e => {
                this.update(e.detail.recordID);
            });
            Accidents.Msg.handleDataChanged(e => {
                this.update(e.detail.recordID);
            });
            this.init();
        }
        init() {
            this.update(Accidents.dataSet.currRecID);
        }
    }
    class ReportPane extends DataPane {
        constructor() {
            super(".rp_rec");
        }
        static getInstance() {
            if (!ReportPane.instance)
                ReportPane.instance = new ReportPane();
            return ReportPane.instance;
        }
        update(recID) {
            const rec = Accidents.dataSet.getRecord(recID);
            this.rootEl.querySelectorAll('.rp_value').forEach(e => {
                e.textContent = "–";
            });
            if (rec)
                for (let name in rec) {
                    const el = this.rootEl.querySelector(`[data-fieldname="${name}"] .rp_value`);
                    if (el)
                        el.textContent = rec[name];
                }
        }
    }
    Accidents.ReportPane = ReportPane;
    class ReportSelPane extends DataPane {
        constructor() {
            super(".rp_buttons");
        }
        static getInstance() {
            if (!ReportSelPane.instance)
                ReportSelPane.instance = new ReportSelPane();
            return ReportSelPane.instance;
        }
        init() {
            this.btnNext = this.rootEl.querySelector("#btn_rec_next");
            this.btnPrev = this.rootEl.querySelector("#btn_rec_prev");
            this.btnNext.addEventListener("click", e => {
                Accidents.dataSet.nextRecord();
            });
            this.btnPrev.addEventListener("click", e => {
                Accidents.dataSet.prevRecord();
            });
            super.init();
        }
        update(recID) {
            const [n, total] = Accidents.dataSet.recordCount(recID);
            this.rootEl.querySelector('.rec_number').textContent = n.toString();
            this.rootEl.querySelector('.rec_total').textContent = total.toString();
            this.btnPrev.disabled = (n <= 1);
            this.btnNext.disabled = (n >= total);
        }
    }
    Accidents.ReportSelPane = ReportSelPane;
})(Accidents || (Accidents = {}));
//# sourceMappingURL=main.js.map